/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	$Id: pgpRPCClientWin32.c,v 1.25.12.1 2001/05/23 17:21:27 hal Exp $
____________________________________________________________________________*/
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <rpc.h>
#include "pgpkeysvc.h"
#include "pgpRPCMsg.h"
#include "pgpPFLErrors.h"
#include "pgpErrors.h"
#include "pgpDebug.h"

static RPC_BINDING_HANDLE Binding;
static PGPInt32 isRPCEnabled = FALSE;
static int pgpRPCCallNativeRPC(char *src, int srclen, char **dst, int *dstlen, unsigned long okayToBlock);
PGPBoolean sIsSDKServiceStarting();
static OSVERSIONINFO osid;
static sCreateRpcBinding();

PGPError
pgpRPCInit()
{
	PGPError err;

	osid.dwOSVersionInfoSize = sizeof(osid);
	GetVersionEx(&osid);

	err = sCreateRpcBinding();
	if (IsPGPError(err))
		return err;

	err = pgpConnect_backRPC();
	if (IsPGPError(err)) {
		PGPInt32 i, max = 5;

		for (i=0; sIsSDKServiceStarting() && (i < max); ++i) {
			Sleep(1000 * (1+i*2));
			err = pgpConnect_backRPC();
			if (err == kPGPError_NoErr) {
				isRPCEnabled = TRUE;
				return kPGPError_NoErr;
			}
		}
		return kPGPError_RPCFailed;
	}

	isRPCEnabled = TRUE;
	return kPGPError_NoErr;
}

void
pgpRPCUninit()
{
	RpcBindingFree(&Binding);
}

	PGPError
PGPsdkReconnect()
{
	PGPError err;

	if (isRPCEnabled == FALSE)
		return kPGPError_NoErr;
	RpcBindingReset(&Binding);
	RpcBindingFree(&Binding);

	err = sCreateRpcBinding();
	if (err != kPGPError_NoErr)
		return err;

	err = pgpReconnect_backRPC();
	return err;
}

	PGPError
sCreateRpcBinding()
{
	unsigned char *pszUuid = NULL;
	unsigned char *pszProtocolSequence;
	unsigned char *pszEndpoint;
	unsigned char *pszNetworkAddress = NULL;
	unsigned char *pszStringBinding = NULL;
	unsigned char *pszOptions = NULL;
    RPC_STATUS status;

	if (osid.dwPlatformId == VER_PLATFORM_WIN32_NT) {
		pszProtocolSequence = "ncacn_np";
		pszEndpoint = "\\pipe\\pgpsdkserv";
	}
	else {
		pszProtocolSequence = "ncalrpc";
		pszEndpoint = "pgpsdkserv";
	}

	status = RpcStringBindingCompose(pszUuid,
										pszProtocolSequence,
										pszNetworkAddress,
										pszEndpoint,
										pszOptions,
										&pszStringBinding);
	if (status) 
		return kPGPError_RPCFailed;

	status = RpcBindingFromStringBinding(pszStringBinding, &Binding);
	if (status)
		return kPGPError_RPCFailed;

	if (osid.dwPlatformId == VER_PLATFORM_WIN32_NT) {
		status = RpcBindingSetAuthInfo(Binding,
					0,
					RPC_C_AUTHN_LEVEL_CONNECT,		// SecurityLevel
					RPC_C_AUTHN_WINNT ,
					0,
					0
					);
		if (status)
			return kPGPError_RPCFailed;
	}

	return kPGPError_NoErr;
}

/*
 * pgpRPCSendPacket
 *	Its is the responsibility of the caller to free up rply_pkt->base.
 *  No memory is allocated when we get a Win32 RPC error, or if the PGP
 *    function status in the message is an error.
 */
PGPError
pgpRPCSendPacket(PGPPackMsg *pkt, PGPPackMsg *rply_pkt, PGPBoolean okayToBlock)
{
	PGPInt32 reply_len, status, length;
	char *reply_buf;
	PGPError err;

	((PGPInt32 *)pkt->base)[2] = pkt->ptr;
	err = pgpRPCCallNativeRPC(pkt->base, pkt->ptr, &reply_buf, &reply_len, okayToBlock);
	PGPFreeData(pkt->base);			/* Free Data-Send Buffer */
	if (IsPGPError(err)) {
		return err;
	}
	if (!okayToBlock && (reply_buf == NULL)) {
		return kPGPError_TLSWouldBlock;
	}
	rply_pkt->base = reply_buf;
	rply_pkt->length = reply_len;
	rply_pkt->ptr = 0;
	status = unpack_int32(rply_pkt);
	length = unpack_int32(rply_pkt);
	if (IsPGPError(status))
		PGPFreeData(reply_buf);
	return status;
}

PGPError
pgpRPCCallNativeRPC(char *src, int srclen, char **dst, int *dstlen, unsigned long okayToBlock)
{
    RPC_STATUS status;
	PGPPackMsg *rm;
	int retval;
	// RPC OUTPUT BUFFER ZONE
    BUFFER Buffer;
    Buffer.BufferLength = 0;
    Buffer.Buffer = NULL;

	/* Sometimes get spurious errors when server is busy */
	do {
		status = GetPGPBuf(Binding, src, srclen, okayToBlock, &Buffer);
	} while ( status == RPC_S_ACCESS_DENIED
			  || status == RPC_S_SERVER_TOO_BUSY );

    if (status != RPC_S_OK)
		return kPGPError_RPCFailed;

	rm = (PGPPackMsg *)Buffer.Buffer;
	*dstlen = Buffer.BufferLength;
	*dst = Buffer.Buffer;
	retval = rm->status;

	return kPGPError_NoErr;
}

void * __RPC_USER MIDL_user_allocate(size_t size)
{
	return PGPNewData(PGPGetDefaultMemoryMgr(), size, 0);
}

void __RPC_USER MIDL_user_free( void *pointer)
{
	PGPFreeData(pointer);
}

PGPBoolean
pgpRPCEnabled()
{
	return isRPCEnabled;
}

static PGPBoolean
sIsSDKServiceStarting()
{
	if (osid.dwPlatformId == VER_PLATFORM_WIN32_NT) {
		SC_HANDLE SCManager, SCService;
		SERVICE_STATUS ServiceStatus;
		BOOL b;

		SCManager = OpenSCManager(NULL, NULL, GENERIC_READ);
		if (SCManager == NULL)
			return FALSE;

		SCService = OpenService(SCManager, "PGPsdkServ", SERVICE_QUERY_STATUS);
		if (SCService == NULL) {
			CloseServiceHandle(SCManager);
			return FALSE;
		}

		b = QueryServiceStatus(SCService, &ServiceStatus);

		CloseServiceHandle(SCService);
		CloseServiceHandle(SCManager);

		if (!b)
			return FALSE;
		if (ServiceStatus.dwCurrentState == SERVICE_START_PENDING)
			return TRUE;
		if (ServiceStatus.dwCurrentState == SERVICE_CONTINUE_PENDING)
			return TRUE;
		if (ServiceStatus.dwCurrentState == SERVICE_RUNNING)
			return TRUE;
		return FALSE;
	}
	else
		return FALSE;
}
